MODULE MAP_Steel_Activity
CONTAINS
! -------------------------------------------------------------------
!
! The procedures in this module calculate the activity of either 
!  iron or carbon in austenite, and also the differentials of these
!  activities.
!
! MODULE contains:
!
!   MAP_STEEL_AFEG
!   MAP_STEEL_DAFEG
!       These procedures calculate activity of IRON in austenite.
!   MAP_STEEL_CG
!   MAP_STEEL_DCG
!       These procedures calculate activity of CARBON 
!         in austenite.
!
!   In  all cases, the parameters accepted by the procedures are:
!
!     T: absolute temperature.
!     W: carbon-carbon interaction energy in austenite (J/mol).
!     X: paraequilibrium carbon concentration of austenite 
!        (mole fraction)-may appear as dummy argument XEQ.
!
!   Procedures in this module also use Modules Global_Data and
!     Free_Energy.
!
! -------------------------------------------------------------------
!
! Function MAP_STEEL_AFEG
!
! H. K. D. H. Bhadeshia, 1984
! Natural logarithm of the activity of iron in austenite
!
! ------------------------------------------------------------------
      FUNCTION MAP_STEEL_AFEG(T,W,XEQ)
        USE MAP_Util_Global_Data
        IMPLICIT NONE
!
! Arguments
!
        REAL(KIND=real_8_30),INTENT(IN):: T,W,XEQ
        REAL(KIND=real_8_30) :: MAP_STEEL_AFEG
!
! Local Variables
!
        REAL(KIND=real_8_30)           :: DEQ,TEQ

        DEQ=SQRT(1-2*(1+2*AJ(T,W))*XEQ+(1+8*AJ(T,W))*XEQ**2)
        TEQ=5.0*LOG((1-XEQ)/(1-2*XEQ))
        TEQ=TEQ+6.0*LOG((1-2*AJ(T,W)+(4*AJ(T,W)-1)*XEQ-DEQ)/&
               (2*AJ(T,W)*(2*XEQ-1)))

        MAP_STEEL_AFEG=TEQ

        RETURN
      END FUNCTION MAP_STEEL_AFEG
! -----------------------------------------------------------------------
! 
! MAP_STEEL_DAFEG DAFEG
!   Function calculating differential of activity of iron in gamma.
!
! -----------------------------------------------------------------------
    FUNCTION MAP_STEEL_DAFEG(T,W,XEQ)
      USE MAP_Util_Global_Data
      IMPLICIT NONE
!
! Arguments
!
      REAL(KIND=real_8_30),INTENT(IN) :: T,W,XEQ
      REAL(KIND=real_8_30) :: MAP_STEEL_DAFEG
!
! Local Variables
!
      REAL(KIND=real_8_30) :: DEQ,ETEQ,ETEQ2

        DEQ=SQRT(1-2*(1+2*AJ(T,W))*XEQ+(1+8*AJ(T,W))*XEQ**2)
        ETEQ=5*((1/(XEQ-1))+2/(1-2*XEQ))
        ETEQ2=6*((4*AJ(T,W)-1-(0.5/DEQ)*(-2-4*AJ(T,W)+2*XEQ+16*XEQ*AJ(T,W))) &
              /(1-2*AJ(T,W)+(4*AJ(T,W)-1)*XEQ-DEQ))+ &
              6*(4*AJ(T,W)/(2*AJ(T,W)*(2*XEQ-1)))

        MAP_STEEL_DAFEG=ETEQ+ETEQ2

      RETURN
    END FUNCTION MAP_STEEL_DAFEG
! -----------------------------------------------------------------------
!
! Function MAP_STEEL_CG
!   Function giving ln(activity) of carbon in 
!   austenite 
!
! -----------------------------------------------------------------------
    FUNCTION MAP_STEEL_CG(T,W,X)
      USE MAP_Util_Global_Data
      IMPLICIT NONE
!
! Arguments
!
      REAL(KIND=real_8_30),INTENT(IN) :: T,W,X
      REAL(KIND=real_8_30) :: MAP_STEEL_CG
!
! Local Variables
!
      REAL(KIND=real_8_30) :: DG,EG

         IF(X .LE. 1.0E-10) THEN
           MAP_STEEL_CG=LOG(1.0E-10)
         ELSE
           DG=SQRT(1.0-2.0*(1.0+2.0*AJ(T,W))*X+(1.0+8.0*AJ(T,W))*X**2)
           EG=5.0*LOG((1.0-2.0*X)/X)+6.0*W/(R*T)+ &
     &     (38575.0-13.48*T)/(R*T)
           MAP_STEEL_CG=EG+6.0E+00*LOG((DG-1+3*X)/(DG+1-3*X))
         ENDIF
      RETURN
    END FUNCTION MAP_STEEL_CG
! -----------------------------------------------------------------------
!
! MAP_STEEL_DCG
!   Function giving differential of ln(activity of carbon in austenite).
!   LFG differential is with respect to X
!
! -----------------------------------------------------------------------
    FUNCTION MAP_STEEL_DCG(T,W,X)
      USE MAP_Util_Global_Data
      IMPLICIT NONE
!
! Arguments
!
      REAL(KIND=real_8_30),INTENT(IN) :: T,W,X
      REAL(KIND=real_8_30) :: MAP_STEEL_DCG
!
! Local Variables
!
      REAL(KIND=real_8_30) DG,DDG

        DG =SQRT(1.0-2.0*(1.0+2.0*AJ(T,W))*X+(1.0+8.0*AJ(T,W))*X**2)
        DDG=(0.5/DG)*(-2.0-4.0*AJ(T,W)+2.0*X+16.0*AJ(T,W)*X)

        MAP_STEEL_DCG=-((10.0/(1.0-2.0*X))+(5.0/X))+6.0*((DDG+3.0) &
     &         /(DG-1.0+3.0*X)-(DDG-3.0)/(DG+1.0-3.0*X))

      RETURN
    END FUNCTION MAP_STEEL_DCG
! -----------------------------------------------------------------------


END MODULE MAP_Steel_Activity
