#include "r.h"
#include "mynr.h"

/* CONJUGATE GRADIENT ALGORITHMS */

/* linmin */

/* #define FABS(x) ( (x>0.0) ? (x) : -(x) ) */
#define FABS(x) fabs(x)

int ncom=0;	/* defining declarations */
double *pcom=0,*xicom=0,(*nrfunc)();

#define TOL linmin_tol

void linmin(p,xi,n,fret,func,linmin_tol)
double p[],xi[],*fret,(*func)(),linmin_tol;
int n;
{
	int j;
	double xx,xmin,fx,fb,fa,bx,ax;
	double brent(),f1dim(),*dvector();
	void mnbrak(),free_dvector();
  static double lastxx = 0.01 ; /* 1.0 might make more sense, but hey! */

	ncom=n;
	pcom=dvector(1,n);
	xicom=dvector(1,n);
	nrfunc=func;
	for (j=1;j<=n;j++) {
		pcom[j]=p[j];
		xicom[j]=xi[j];
	}
  fa = *fret ; /* 21 05 94: pass the current value through to mnbrak */
	ax=0.0;
  xx=lastxx ;
  bx=2.0 * xx ;
	
	mnbrak(&ax,&xx,&bx,&fa,&fx,&fb,f1dim);

  lastxx = fabs ( xx ) ;
  if ( fabs ( ax ) > lastxx ) lastxx =  fabs ( ax ) * 0.5 ;

	*fret=brent(ax,xx,bx,f1dim,TOL,&xmin);
	for (j=1;j<=n;j++) {
		xi[j] *= xmin;
		p[j] += xi[j];
	}
	free_dvector(xicom,1,n);
	free_dvector(pcom,1,n);
}

#undef TOL


/* dlinmin.c */

#define TOL 2.0e-4

	/* defining declarations */
void (*nrdfun)();

void dlinmin(p,xi,n,fret,func,dfunc)
double p[],xi[],*fret,(*func)();
void (*dfunc)();
int n;
{
	int j;
	double xx,xmin,fx,fb,fa,bx,ax;
	double dbrent(),f1dim(),df1dim(),*dvector();
	void mnbrak(),free_dvector();

	ncom=n;
	pcom=dvector(1,n);
	xicom=dvector(1,n);
	nrfunc=func;
	nrdfun=dfunc;
	for (j=1;j<=n;j++) {
		pcom[j]=p[j];
		xicom[j]=xi[j];
	}
	ax=0.0;
	xx=1.0;
	bx=2.0;
	mnbrak(&ax,&xx,&bx,&fa,&fx,&fb,f1dim);
	*fret=dbrent(ax,xx,bx,f1dim,df1dim,TOL,&xmin);
	for (j=1;j<=n;j++) {
		xi[j] *= xmin;
		p[j] += xi[j];
	}
	free_dvector(xicom,1,n);
	free_dvector(pcom,1,n);
}

#undef TOL

/* brent.c */

#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SIGN(a,b) ((b) > 0.0 ? FABS(a) : -FABS(a))
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

double brent(ax,bx,cx,f,tol,xmin)
double ax,bx,cx,tol,*xmin;
double (*f)();	/* ANSI: double (*f)(double); */
{
	int iter;
	double a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	double e=0.0;
	void nrerror();

	a=((ax < cx) ? ax : cx);
	b=((ax > cx) ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*FABS(x)+ZEPS);
		if (FABS(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
/* 			printf("brent: %g %g %g %g %g\n",FABS(x-xm), (tol2-0.5*(b-a)),tol2,x,xm ); */
			return fx;
		}
		if (FABS(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=FABS(q);
			etemp=e;
			e=d;
			if (FABS(p) >= FABS(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(FABS(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		fu=(*f)(u);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in BRENT");
	*xmin=x;
	return fx;
}


double brent2(ax,bx,cx,f,f_arg,tol,xmin)
double ax,bx,cx,tol,*xmin;
double (*f)();	/* ANSI: double (*f)(double); */
void *f_arg;
{
	int iter;
	double a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	double e=0.0;
	void nrerror();

	a=((ax < cx) ? ax : cx);
	b=((ax > cx) ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x,f_arg);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*FABS(x)+ZEPS);
		if (FABS(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (FABS(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=FABS(q);
			etemp=e;
			e=d;
			if (FABS(p) >= FABS(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(FABS(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		fu=(*f)(u,f_arg);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in BRENT");
	*xmin=x;
	return fx;
}

#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SIGN

/* dbrent.c */

#define ITMAX 100
#define ZEPS 1.0e-10
#define SIGN(a,b) ((b) > 0.0 ? FABS(a) : -FABS(a))
#define MOV3(a,b,c, d,e,f) (a)=(d);(b)=(e);(c)=(f);

double dbrent(ax,bx,cx,f,df,tol,xmin)
double ax,bx,cx,tol,*xmin;
double (*f)(),(*df)(); /* ANSI: double (*f)(double),(*df)(double); */
{
	int iter,ok1,ok2;
	double a,b,d,d1,d2,du,dv,dw,dx,e=0.0;
	double fu,fv,fw,fx,olde,tol1,tol2,u,u1,u2,v,w,x,xm;
	void nrerror();

	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=(*f)(x);
	dw=dv=dx=(*df)(x);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol1=tol*FABS(x)+ZEPS;
		tol2=2.0*tol1;
		if (FABS(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (FABS(e) > tol1) {
			d1=2.0*(b-a);
			d2=d1;
			if (dw != dx)  d1=(w-x)*dx/(dx-dw);
			if (dv != dx)  d2=(v-x)*dx/(dx-dv);
			u1=x+d1;
			u2=x+d2;
			ok1 = (a-u1)*(u1-b) > 0.0 && dx*d1 <= 0.0;
			ok2 = (a-u2)*(u2-b) > 0.0 && dx*d2 <= 0.0;
			olde=e;
			e=d;
			if (ok1 || ok2) {
				if (ok1 && ok2)
					d=(FABS(d1) < FABS(d2) ? d1 : d2);
				else if (ok1)
					d=d1;
				else
					d=d2;
				if (FABS(d) <= FABS(0.5*olde)) {
					u=x+d;
					if (u-a < tol2 || b-u < tol2)
						d=SIGN(tol1,xm-x);
				} else {
					d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
				}
			} else {
				d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
			}
		} else {
			d=0.5*(e=(dx >= 0.0 ? a-x : b-x));
		}
		if (FABS(d) >= tol1) {
			u=x+d;
			fu=(*f)(u);
		} else {
			u=x+SIGN(tol1,d);
			fu=(*f)(u);
			if (fu > fx) {
				*xmin=x;
				return fx;
			}
		}
		du=(*df)(u);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			MOV3(v,fv,dv, w,fw,dw)
			MOV3(w,fw,dw, x,fx,dx)
			MOV3(x,fx,dx, u,fu,du)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				MOV3(v,fv,dv, w,fw,dw)
				MOV3(w,fw,dw, u,fu,du)
			} else if (fu < fv || v == x || v == w) {
				MOV3(v,fv,dv, u,fu,du)
			}
		}
	}
	nrerror("Too many iterations in routine DBRENT");
}

#undef ITMAX
#undef ZEPS
#undef SIGN
#undef MOV3

/* frprmn.c */

#define ITMAX 400
#define EPS 1.0e-10
#define FREEALL free_dvector(xi,1,n);free_dvector(h,1,n);free_dvector(g,1,n);

void frprmn(p,n,ftol,iter,fret,func,dfunc,flinmintol)
double p[],ftol,*fret,(*func)(),flinmintol;
void (*dfunc)();
int n,*iter;
{
	int j,its;
	double gg,gam,fp,dgg;
	double *g,*h,*xi,*dvector();
	void linmin(),nrerror(),free_dvector();

	g=dvector(1,n);
	h=dvector(1,n);
	xi=dvector(1,n);
	fp=(*func)(p);
	(*dfunc)(p,xi);
	for (j=1;j<=n;j++) {
		g[j] = -xi[j];
		xi[j]=h[j]=g[j];
	}
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
    *fret = fp ; /* 21 05 94: pass the current value through to mnbrak */
		linmin(p,xi,n,fret,func,flinmintol); /* NB linmin has an extra arg */
		if (2.0*FABS(*fret-fp) <= ftol*(FABS(*fret)+FABS(fp)+EPS)) {
			FREEALL
			return;
		}
		fp=(*func)(p);
		(*dfunc)(p,xi);
		dgg=gg=0.0;
		for (j=1;j<=n;j++) {
			gg += g[j]*g[j];
/*		  dgg += xi[j]*xi[j];	*/
			dgg += (xi[j]+g[j])*xi[j];
		}
		if (gg == 0.0) {
			FREEALL
			return;
		}
		gam=dgg/gg;
		for (j=1;j<=n;j++) {
			g[j] = -xi[j];
			xi[j]=h[j]=g[j]+gam*h[j];
		}
	}
	printf("Too many iterations in FRPRMN, but continuing.\n"); 
	FREEALL	
	return;
}

#undef ITMAX
#undef EPS
#undef FREEALL


/* frprmn_a.c */
/* Modified to use absolute tolerance for outer loop. 
NB THe inner loop measures a tolerance in parameter location, 
not in function value!  */

#define ITMAX 400
#define EPS 1.0e-10
#define FREEALL free_dvector(xi,1,n);free_dvector(h,1,n);free_dvector(g,1,n);

void frprmn_a(p,n,abtol,iter,fret,func,dfunc,flinmintol)
double p[],abtol,*fret,(*func)(),flinmintol;
void (*dfunc)();
int n,*iter;
{
	int j,its;
	double gg,gam,fp,dgg;
	double *g,*h,*xi,*dvector();
	void linmin(),nrerror(),free_dvector();

	g=dvector(1,n);
	h=dvector(1,n);
	xi=dvector(1,n);
	fp=(*func)(p);
	(*dfunc)(p,xi);
	for (j=1;j<=n;j++) {
		g[j] = -xi[j];
		xi[j]=h[j]=g[j];
	}
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
    *fret = fp ; /* 21 05 94: pass the current value through to mnbrak */
		linmin(p,xi,n,fret,func,flinmintol); 
		if (FABS(*fret-fp) <= abtol) {
/*		        printf("returning at %f %f %f\n",*fret,fp,abtol); */
			FREEALL
			return;
		}
		fp=(*func)(p);
		(*dfunc)(p,xi);
		dgg=gg=0.0;
		for (j=1;j<=n;j++) {
			gg += g[j]*g[j];
/*		  dgg += xi[j]*xi[j];	*/
			dgg += (xi[j]+g[j])*xi[j];
		}
		if (gg == 0.0) {
			FREEALL
			return;
		}
		gam=dgg/gg;
		for (j=1;j<=n;j++) {
			g[j] = -xi[j];
			xi[j]=h[j]=g[j]+gam*h[j];
		}
	}
	printf("Too many iterations in FRPRMN, but continuing.\n"); 
	FREEALL	
	return;
}

#undef ITMAX
#undef EPS
#undef FREEALL

/* mnbrak.c */

#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define SIGN(a,b) ((b) > 0.0 ? FABS(a) : -FABS(a))
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);
#define MAX(a,b) ((a) > (b) ? (a) : (b) ) 

void mnbrak(ax,bx,cx,fa,fb,fc,func)
double *ax,*bx,*cx,*fa,*fb,*fc;
double (*func)();	/* ANSI: double (*func)(double); */
{
  double ulim,u,r,q,fu,dum;

/*  printf("mnbrak: %7.4g ?= ", *fa ); */
/*  *fa=(*func)(*ax);    this value is received always in my applications */
/*  printf("%7.4g\n", *fa ); */
  *fb=(*func)(*bx);
  if (*fb > *fa) {
    SHFT(dum,*ax,*bx,dum)
      SHFT(dum,*fb,*fa,dum)
  }
	*cx=(*bx)+GOLD*(*bx-*ax);
	*fc=(*func)(*cx);
	while (*fb > *fc) {
		r=(*bx-*ax)*(*fb-*fc);
		q=(*bx-*cx)*(*fb-*fa);
		u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/
			(2.0*SIGN(MAX(FABS(q-r),TINY),q-r));
		ulim=(*bx)+GLIMIT*(*cx-*bx);
		if ((*bx-u)*(u-*cx) > 0.0) {
			fu=(*func)(u);
			if (fu < *fc) {
				*ax=(*bx);
				*bx=u;
				*fa=(*fb);
				*fb=fu;
				return;
			} else if (fu > *fb) {
				*cx=u;
				*fc=fu;
				return;
			}
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u);
		} else if ((*cx-u)*(u-ulim) > 0.0) {
			fu=(*func)(u);
			if (fu < *fc) {
				SHFT(*bx,*cx,u,*cx+GOLD*(*cx-*bx))
				SHFT(*fb,*fc,fu,(*func)(u))
			}
		} else if ((u-ulim)*(ulim-*cx) >= 0.0) {
			u=ulim;
			fu=(*func)(u);
		} else {
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u);
		}
		SHFT(*ax,*bx,*cx,u)
		SHFT(*fa,*fb,*fc,fu)
	}
}

void mnbrak2(ax,bx,cx,fa,fb,fc,func,f_arg)
double *ax,*bx,*cx,*fa,*fb,*fc;
double (*func)();	/* ANSI: double (*func)(double); */
void *f_arg;
{
	double ulim,u,r,q,fu,dum;

/*  *fa = ( *func ) ( *ax,f_arg ) ; */
/* I have cut this because in my applications, func has been evaluated just */
/* I pass this value instead */
	*fb=(*func)(*bx,f_arg);
	if (*fb > *fa) {
		SHFT(dum,*ax,*bx,dum)
		SHFT(dum,*fb,*fa,dum)
	}
	*cx=(*bx)+GOLD*(*bx-*ax);
	*fc=(*func)(*cx,f_arg);
	while (*fb > *fc) {
		r=(*bx-*ax)*(*fb-*fc);
		q=(*bx-*cx)*(*fb-*fa);
		u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/
			(2.0*SIGN(MAX(FABS(q-r),TINY),q-r));
		ulim=(*bx)+GLIMIT*(*cx-*bx);
		if ((*bx-u)*(u-*cx) > 0.0) {
			fu=(*func)(u,f_arg);
			if (fu < *fc) {
				*ax=(*bx);
				*bx=u;
				*fa=(*fb);
				*fb=fu;
				return;
			} else if (fu > *fb) {
				*cx=u;
				*fc=fu;
				return;
			}
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u,f_arg);
		} else if ((*cx-u)*(u-ulim) > 0.0) {
			fu=(*func)(u,f_arg);
			if (fu < *fc) {
				SHFT(*bx,*cx,u,*cx+GOLD*(*cx-*bx))
				SHFT(*fb,*fc,fu,(*func)(u,f_arg))
			}
		} else if ((u-ulim)*(ulim-*cx) >= 0.0) {
			u=ulim;
			fu=(*func)(u,f_arg);
		} else {
			u=(*cx)+GOLD*(*cx-*bx);
			fu=(*func)(u,f_arg);
		}
		SHFT(*ax,*bx,*cx,u)
		SHFT(*fa,*fb,*fc,fu)
	}
}

#undef GOLD
#undef GLIMIT
#undef TINY
#undef MAX
#undef SIGN
#undef SHFT

extern int ncom;        /* defined in LINMIN */
extern double *pcom,*xicom,(*nrfunc)();

double f1dim(x) 
double x;
{
        int j;
        double f,*xt,*dvector();
        void free_dvector();

        xt=dvector(1,ncom);
        for (j=1;j<=ncom;j++) xt[j]=pcom[j]+x*xicom[j];
        f=(*nrfunc)(xt);
        free_dvector(xt,1,ncom);
        return f;
}

double f1dim2(x,f_argp) 
double x;
void *f_argp;
{
        int j;
        double f,*xt,*dvector();
        void free_dvector();
	f1dim2_arg f_arg;

	f_arg = *( (f1dim2_arg *)f_argp);
        xt=dvector(1,f_arg.n);
        for (j=1;j<=f_arg.n;j++) xt[j]=f_arg.p[j]+x*f_arg.xi[j];
        f=(*f_arg.nfunc)(xt,f_arg.func_arg);
        free_dvector(xt,1,f_arg.n);
        return f;
}

extern int ncom;	/* defined in DLINMIN */
extern double *pcom,*xicom,(*nrfunc)();
extern void (*nrdfun)();

double df1dim(x)
double x;
{
	int j;
	double df1=0.0;
	double *xt,*df,*dvector();
	void free_dvector();

	xt=dvector(1,ncom);
	df=dvector(1,ncom);
	for (j=1;j<=ncom;j++) xt[j]=pcom[j]+x*xicom[j];
	(*nrdfun)(xt,df);
	for (j=1;j<=ncom;j++) df1 += df[j]*xicom[j];
	free_dvector(df,1,ncom);
	free_dvector(xt,1,ncom);
	return df1;
}

/* dfpmin.c */
/* Modified 30 Nov 90 so that it uses allocated space for the Hessian */  
/* 	hessin=matrix(1,n,1,n);	must be declared and initialised previously */
/* also modified to force it to do at least *iter loops, (so that a good hessian 
estimate is made) */ 
/* also modified to pass a tolerance to linmin when it is called */

#define ITMAX 400
#define EPS 1.0e-10

void dfpmin(p,n,ftol,iter,fret,func,dfunc,hessin,frac_lin_tol)
double p[],ftol,*fret,(*func)(),**hessin,frac_lin_tol;
void (*dfunc)();
int n,*iter;
{
  int j,i,its;
  double fp,fae,fad,fac;
  double *xi,*g,*dg,*hdg,*dvector();
  void linmin(),nrerror(),free_dvector();
  int	min_its;
  
  min_its= *iter; 
  
  xi=dvector(1,n);
  g=dvector(1,n);
  dg=dvector(1,n);
  hdg=dvector(1,n);
  fp=(*func)(p);
  (*dfunc)(p,g);
/*	for (i=1;i<=n;i++) {
		for (j=1;j<=n;j++) hessin[i][j]=0.0;
		hessin[i][i]=1.0; 
		xi[i] = -g[i];
	} */
  for (i=1;i<=n;i++) {
    xi[i]=0.0;
    for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
  }
  for (its=1;its<=ITMAX;its++) {
    *iter=its;
    *fret = fp ; /* 21 05 94: pass the current value through to mnbrak */
	
    linmin(p,xi,n,fret,func,frac_lin_tol);
    if ((its>=min_its)&&(2.0*FABS(*fret-fp) <= ftol*(FABS(*fret)+FABS(fp)+EPS))) {
      free_dvector(hdg,1,n);
      free_dvector(dg,1,n);
      free_dvector(g,1,n);
      free_dvector(xi,1,n);
      return;
    }
    fp=(*fret);
    for (i=1;i<=n;i++) dg[i]=g[i];
    *fret=(*func)(p);
    (*dfunc)(p,g);
    for (i=1;i<=n;i++) dg[i]=g[i]-dg[i];
    for (i=1;i<=n;i++) {
      hdg[i]=0.0;
      for (j=1;j<=n;j++) hdg[i] += hessin[i][j]*dg[j];
    }
    fac=fae=0.0;
    for (i=1;i<=n;i++) {
      fac += dg[i]*xi[i];
      fae += dg[i]*hdg[i];
    }
    fac=1.0/fac;
    fad=1.0/fae;
    for (i=1;i<=n;i++) dg[i]=fac*xi[i]-fad*hdg[i];
    for (i=1;i<=n;i++)
      for (j=1;j<=n;j++)
	hessin[i][j] += fac*xi[i]*xi[j]
	  -fad*hdg[i]*hdg[j]+fae*dg[i]*dg[j];
    for (i=1;i<=n;i++) {
      xi[i]=0.0;
      for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
    }
  }
/*	nrerror("Too many iterations in DFPMIN"); */
  printf("Too many iterations in DFPMIN, but continuing.\n"); 
  free_dvector(hdg,1,n);
  free_dvector(dg,1,n);
  free_dvector(g,1,n);
  free_dvector(xi,1,n);
  return;
}

#undef ITMAX
#undef EPS

/* dfpmin_a.c */
/* Modified 30 Nov 90 so that it uses allocated space for the Hessian */  
/* 	hessin=matrix(1,n,1,n);	must be declared and initialised previously */
/* also modified to force it to do at least *iter loops, (so that a good hessian 
estimate is made) */ 
/* also modified to pass a tolerance to linmin when it is called */
/* "_a" includes absolute tolerance condition for the overall minimisation */

#define ITMAX 400

void dfpmin_a(p,n,abtol,iter,fret,func,dfunc,hessin,frac_lin_tol)
double p[],abtol,*fret,(*func)(),**hessin,frac_lin_tol;
void (*dfunc)();
int n,*iter;
{
	int j,i,its;
	double fp,fae,fad,fac;
	double *xi,*g,*dg,*hdg,*dvector();
	void linmin(),nrerror(),free_dvector();
	int	min_its;

	min_its= *iter; 

	xi=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);
	hdg=dvector(1,n);
	fp=(*func)(p);
	(*dfunc)(p,g);
	for (i=1;i<=n;i++) {
		xi[i]=0.0;
		for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
	}
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
    *fret = fp ; /* 21 05 94: pass the current value through to mnbrak */
		linmin(p,xi,n,fret,func,frac_lin_tol);
		if ((its>=min_its)&&(2.0*FABS(*fret-fp) <= abtol) ) {
			free_dvector(hdg,1,n);
			free_dvector(dg,1,n);
			free_dvector(g,1,n);
			free_dvector(xi,1,n);
			return;
		}
		fp=(*fret);
		for (i=1;i<=n;i++) dg[i]=g[i];
		*fret=(*func)(p);
		(*dfunc)(p,g);
		for (i=1;i<=n;i++) dg[i]=g[i]-dg[i];
		for (i=1;i<=n;i++) {
			hdg[i]=0.0;
			for (j=1;j<=n;j++) hdg[i] += hessin[i][j]*dg[j];
		}
		fac=fae=0.0;
		for (i=1;i<=n;i++) {
			fac += dg[i]*xi[i];
			fae += dg[i]*hdg[i];
		}
		fac=1.0/fac;
		fad=1.0/fae;
		for (i=1;i<=n;i++) dg[i]=fac*xi[i]-fad*hdg[i];
		for (i=1;i<=n;i++)
			for (j=1;j<=n;j++)
				hessin[i][j] += fac*xi[i]*xi[j]
					-fad*hdg[i]*hdg[j]+fae*dg[i]*dg[j];
		for (i=1;i<=n;i++) {
			xi[i]=0.0;
			for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
		}
	}
/*	nrerror("Too many iterations in DFPMIN"); */
	printf("Too many iterations in DFPMIN, but continuing.\n"); 
			free_dvector(hdg,1,n);
			free_dvector(dg,1,n);
			free_dvector(g,1,n);
			free_dvector(xi,1,n);
			return;
}

#undef ITMAX

/* dfpmin2.c */
/* Modified 30 Nov 90 so that it uses allocated space for the Hessian */  
/* 	hessin=matrix(1,n,1,n);	must be declared and initialised previously */
/* also modified to force it to do at least *iter loops, (so that a good hessian 
estimate is made) */ 
/* also modified to pass a tolerance to linmin when it is called */
/* Modified 16 3 92 from dfpmin.c so that the functions func and dfunc both 
   include an additional structure argument that contains all the other variables */
/* So that globals are no longer needed */

#define ITMAX 400
#define EPS 1.0e-10

void dfpmin2(p,n,ftol,iter,fret,func,func_arg,dfunc,dfunc_arg,hessin,frac_lin_tol)
double p[],ftol,*fret,(*func)(),**hessin,frac_lin_tol;
void (*dfunc)();
void *func_arg,*dfunc_arg;
int n,*iter;
{
	int j,i,its;
	double fp,fae,fad,fac;
	double *xi,*g,*dg,*hdg,*dvector();
	void linmin2(),nrerror(),free_dvector();
	int	min_its;

	min_its= *iter; 

	xi=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);
	hdg=dvector(1,n);
	fp=(*func)(p,func_arg);
	(*dfunc)(p,g,dfunc_arg);

	for (i=1;i<=n;i++) {
		xi[i]=0.0;
		for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
	}
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
		linmin2(p,xi,n,fret,func,func_arg,frac_lin_tol);
		if ((its>=min_its)&&(2.0*FABS(*fret-fp) <= ftol*(FABS(*fret)+FABS(fp)+EPS))) {
			free_dvector(hdg,1,n);
			free_dvector(dg,1,n);
			free_dvector(g,1,n);
			free_dvector(xi,1,n);
			return;
		}
		fp=(*fret);
		for (i=1;i<=n;i++) dg[i]=g[i];
		*fret=(*func)(p,func_arg);
		(*dfunc)(p,g,dfunc_arg);
		for (i=1;i<=n;i++) dg[i]=g[i]-dg[i];
		for (i=1;i<=n;i++) {
			hdg[i]=0.0;
			for (j=1;j<=n;j++) hdg[i] += hessin[i][j]*dg[j];
		}
		fac=fae=0.0;
		for (i=1;i<=n;i++) {
			fac += dg[i]*xi[i];
			fae += dg[i]*hdg[i];
		}
		fac=1.0/fac;
		fad=1.0/fae;
		for (i=1;i<=n;i++) dg[i]=fac*xi[i]-fad*hdg[i];
		for (i=1;i<=n;i++)
			for (j=1;j<=n;j++)
				hessin[i][j] += fac*xi[i]*xi[j]
					-fad*hdg[i]*hdg[j]+fae*dg[i]*dg[j];
		for (i=1;i<=n;i++) {
			xi[i]=0.0;
			for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
		}
	}
/*	nrerror("Too many iterations in DFPMIN2"); */
	printf("Too many iterations in DFPMIN2, but continuing.\n"); 
			free_dvector(hdg,1,n);
			free_dvector(dg,1,n);
			free_dvector(g,1,n);
			free_dvector(xi,1,n);
			return;
}

#undef ITMAX
#undef EPS

/* Now here is linmin2, which goes with dfpmin2 */

#define TOL linmin_tol

void linmin2(p,xi,n,fret,func,func_arg,linmin_tol)
double p[],xi[],*fret,(*func)(),linmin_tol;
void *func_arg;
int n;
{
	int j;
	double xx,xmin,fx,fb,fa,bx,ax;
	double brent(),f1dim(),*dvector();
	void mnbrak(),free_dvector();
  static double lastxx = 0.01 ; /* 1.0 might make more sense, but hey! */
	f1dim2_arg f_arg;

/*	ncom=n; */
/*	pcom=dvector(1,n);*/ /* These are no longer externals */
/*	xicom=dvector(1,n); */
/* nrfunc is not used by f1dim2 	nrfunc=func;  */
	f_arg.n=n;
	f_arg.p=p;
	f_arg.xi=xi;
	f_arg.nfunc=func;
	f_arg.func_arg = func_arg;
/* typedef struct {
  int n;
  double *p,*xi;
  double (*nfunc)();
  void *func_arg;
} f1dim2_arg; */
	for (j=1;j<=n;j++) {
		f_arg.p[j]=p[j];
		f_arg.xi[j]=xi[j];
	}
  fa = *fret ; /* 21 05 94: pass the current value through to mnbrak */
	ax=0.0;
  xx=lastxx ;
  bx=2.0 * xx ;
	mnbrak2(&ax,&xx,&bx,&fa,&fx,&fb,f1dim2,&f_arg); 
  lastxx = fabs ( xx ) ;
  if ( fabs ( ax ) > lastxx ) lastxx =  fabs ( ax ) * 0.5 ;
	*fret=brent2(ax,xx,bx,f1dim2,&f_arg,TOL,&xmin);
	for (j=1;j<=n;j++) {
		xi[j] *= xmin;
		p[j] += xi[j];
	}
}
#undef TOL

#define EPS 1.0e-10

/* to make a better estimate of the hessin -- program actually generates 
the Hessian directly from derivative differences */
/* this routine makes use of the diagonal elements in the hessin that it is fed */ 
/* the returned value is the determinant of the hessin */ 

double improve_hessin(p,n,func,dfunc,hessin,verbose,kappa)
double p[],(*func)(),**hessin,kappa;
void (*dfunc)();
int n,verbose; /* suggest kappa=0.01 but for backprop it seemed to work down to exp-28!!! */
{
	int 	j,i;
	double	**hessian,*g,*dg,*solution,h,d;

	hessian=dmatrix(1,n,1,n);
	solution=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);

	for (i=1;i<=n;i++) 
		solution[i]=p[i];
	
	for (i=1;i<=n;i++){
		p[i] += (h=kappa* sqrt(hessin[i][i])); 
		if(verbose>1) printf("perturbation = %g\n",h);
		(*dfunc)(p,dg);
		p[i] -= 2.0*h; 
		(*dfunc)(p,g);
		for (j=1;j<=n;j++) hessian[i][j]=(dg[j]-g[j])/(h*2.0);
		p[i] = solution[i];
	}

	if(verbose){
		printf("Hessian before symmetrised:\n"); 
		printoutdmatrix(hessian,1,n,1,n,verbose); 
	}
	symmetrise_dmatrix(hessian,n);
	d = invert_dmatrix(hessian,n,hessin);

	if(verbose){
		pause_for_return();
		printf("Hessin before symmetrised:\n"); 
		printoutdmatrix(hessin,1,n,1,n,verbose); 
	}
	free_dmatrix(hessian,1,n,1,n);
	free_dvector(dg,1,n);
	free_dvector(g,1,n);
	free_dvector(solution,1,n);
	return (1.0/d);
}

/* Very messy modified version that also uses tridag etc to find the eigenvalues 
and clip them if they are too small. The determinant and trace of hessin 
evaluated this way are returned in alt_det, alt_tr */
double improve_hessin3(p,n,func,dfunc,hessin,verbose,kappa,mineig,alt_det_p,alt_tr_p)
double p[],(*func)(),**hessin,kappa,mineig,*alt_det_p,*alt_tr_p;
void (*dfunc)();
int n,verbose; 
{
	int 	j,i,clipped;
	double	**hessian,*g,*dg,*solution,h,d,*lambda;

	hessian=dmatrix(1,n,1,n);
	solution=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);
	lambda=dvector(1,n);

	for (i=1;i<=n;i++) 
		solution[i]=p[i];
	
	for (i=1;i<=n;i++){
		p[i] += (h=kappa* sqrt(hessin[i][i])); 
		if(verbose>1) printf("perturbation = %g\n",h);
		(*dfunc)(p,dg);
		p[i] -= 2.0*h; 
		(*dfunc)(p,g);
		for (j=1;j<=n;j++) hessian[i][j]=(dg[j]-g[j])/(h*2.0);
		p[i] = solution[i];
	}

	if(verbose){
		printf("Hessian before symmetrised:\n"); 
		printoutdmatrix(hessian,1,n,1,n,verbose); 
	}
	symmetrise_dmatrix(hessian,n);
	d = invert_dmatrix(hessian,n,hessin);
	find_eigs(hessian,n,lambda);
	clipped=clip_eigs(lambda,n,mineig);
	printf("Number of clipped eigs = %d\n",clipped);
	det_and_tr_from_eigs(lambda,n,alt_det_p,alt_tr_p);
	
	if(verbose){
		pause_for_return();
		printf("Hessin before symmetrised:\n"); 
		printoutdmatrix(hessin,1,n,1,n,verbose); 
	}
	free_dmatrix(hessian,1,n,1,n);
	free_dvector(dg,1,n);
	free_dvector(g,1,n);
	free_dvector(solution,1,n);
	free_dvector(lambda,1,n);
	return (1.0/d);
}

/* Very messy modified version that also uses tridag etc to find the eigenvalues 
and clip them if they are too small. The determinant and trace of hessin 
evaluated this way are returned in alt_det, alt_tr */
double luimprove_hessin(p,n,func,dfunc,hessin,verbose,kappa,mineig,alt_det_p,alt_tr_p,luhessin,luhessinindx,scratchv,hessian)
double p[],(*func)(),**hessin,**hessian,**luhessin,*scratchv,kappa,mineig,*alt_det_p,*alt_tr_p;
void (*dfunc)();
int n,verbose,*luhessinindx; 
{
	int 	j,i,clipped;
	double	*g,*dg,*solution,h,d,*lambda;

	solution=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);
	lambda=dvector(1,n);

	for (i=1;i<=n;i++) 
		solution[i]=p[i];
	
	for (i=1;i<=n;i++){
		p[i] += (h=kappa* sqrt(hessin[i][i])); 
		if(verbose>1) printf("perturbation = %g\n",h);
		(*dfunc)(p,dg);
		p[i] -= 2.0*h; 
		(*dfunc)(p,g);
		for (j=1;j<=n;j++) hessian[i][j]=(dg[j]-g[j])/(h*2.0);
		p[i] = solution[i];
	}

	if(verbose){
		printf("Hessian before symmetrised:\n"); 
		printoutdmatrix(hessian,1,n,1,n,verbose); 
	}
	symmetrise_dmatrix(hessian,n);
	d = luinvert_dmatrix(hessian,n,hessin,luhessin,luhessinindx,scratchv);
	find_eigs(hessian,n,lambda);
	clipped=clip_eigs(lambda,n,mineig);
	printf("Number of clipped eigs = %d\n",clipped);
	det_and_tr_from_eigs(lambda,n,alt_det_p,alt_tr_p);
	
	if(verbose){
		pause_for_return();
		printf("Hessin before symmetrised:\n"); 
		printoutdmatrix(hessin,1,n,1,n,verbose); 
	}
	free_dvector(dg,1,n);
	free_dvector(g,1,n);
	free_dvector(solution,1,n);
	free_dvector(lambda,1,n);
	return (1.0/d);
}

/* INCOMPLETE program based on dfpmin to make a better estimate of the hessin */
/* before calling this routine it should be confirmed that the current 
	hessin is not too singular */ 
void improve_hessin2(p,n,ftol,iter,fret,func,dfunc,hessin)
double p[],ftol,*fret,(*func)(),**hessin;
void (*dfunc)();
int n,*iter; /* iter is the number of iterations to do */
{
	int j,i,its;
	double fae,fad,fac;
	double *xi,*g,*dg,*hdg,*dvector();
	void linmin(),nrerror(),free_dvector();

	xi=dvector(1,n);
	g=dvector(1,n);
	dg=dvector(1,n);
	hdg=dvector(1,n);
	(*dfunc)(p,g);
	for (i=1;i<=n;i++) {
		xi[i]=0.0;
		for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
	}
	for (its=1;its<=*iter;its++) {
		for (i=1;i<=n;i++) dg[i]=g[i];
/* pick a displacement x, with magnitude small enough that higher terms don't kick in */ 
		for (j=1;j<=n;j++) {
			p[j] += xi[j];
		}
		(*dfunc)(p,g);
		for (i=1;i<=n;i++) dg[i]=g[i]-dg[i];
		for (i=1;i<=n;i++) {
			hdg[i]=0.0;
			for (j=1;j<=n;j++) hdg[i] += hessin[i][j]*dg[j];
		}
		fac=fae=0.0;
		for (i=1;i<=n;i++) {
			fac += dg[i]*xi[i];
			fae += dg[i]*hdg[i];
		}
		fac=1.0/fac;
		fad=1.0/fae;
		for (i=1;i<=n;i++) dg[i]=fac*xi[i]-fad*hdg[i];
		for (i=1;i<=n;i++)
			for (j=1;j<=n;j++)
				hessin[i][j] += fac*xi[i]*xi[j]
					-fad*hdg[i]*hdg[j]+fae*dg[i]*dg[j];
		for (i=1;i<=n;i++) {
			xi[i]=0.0;
			for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
		}
	}
	free_dvector(hdg,1,n);
	free_dvector(dg,1,n);
	free_dvector(g,1,n);
	free_dvector(xi,1,n);
	return;
}

#undef EPS

#undef FABS











